<?php
// Include the database configuration file and header
include 'config.php';
include 'header.php';

// Start session
session_start();

// Check if the user is logged in and has status 2
if (!isset($_SESSION['user_id']) || $_SESSION['status'] != 2) {
    header('Location: login.php');
    exit;
}

// Check if ID is provided in the URL
if (!isset($_GET['id'])) {
    die("User ID not specified.");
}

$userId = intval($_GET['id']);

// Initialize success flag
$updateSuccess = false;

// Fetch user details from the database
try {
    $sql = "SELECT u.id, u.username, u.password, u.email, u.contact_number, u.full_name, u.mt5username, 
                   u.status, COALESCE(u.investment, 0) AS investment, COALESCE(ui.balance, 0) AS balance,
                   COALESCE(ui.equity, 0) AS equity, COALESCE(ui.profit, 0) AS profit,
                   COALESCE(ui.margin, 0) AS margin, COALESCE(ui.open_trades, '') AS open_trades, 
                   COALESCE(ui.timestamp, u.timestamp) AS timestamp, COALESCE(ui.online, u.online) AS online,
                   COALESCE(u.percentage, 0) AS percentage
            FROM users u
            LEFT JOIN userinfo ui ON u.mt5username = ui.mt5username
            WHERE u.id = :id";

    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':id', $userId, PDO::PARAM_INT);
    $stmt->execute();

    // Fetch the result
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        die("User not found.");
    }

} catch (PDOException $e) {
    echo 'Query failed: ' . $e->getMessage();
    exit; // Stop the script if query fails
}

// Handle form submission for editing user details
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Sanitize and validate input
    $username = htmlspecialchars(trim($_POST['username']));
    $email = htmlspecialchars(trim($_POST['email']));
    $contactNumber = htmlspecialchars(trim($_POST['contact_number']));
    $fullName = htmlspecialchars(trim($_POST['full_name']));
    $investment = floatval($_POST['investment']);
    $percentage = floatval($_POST['percentage']);

    // Prepare SQL UPDATE statement
    try {
        $updateSql = "UPDATE users 
                      SET username = :username, email = :email, contact_number = :contact_number, 
                          full_name = :full_name, investment = :investment, percentage = :percentage
                      WHERE mt5username = :mt5username";
        
        $updateStmt = $pdo->prepare($updateSql);
        $updateStmt->bindParam(':username', $username);
        $updateStmt->bindParam(':email', $email);
        $updateStmt->bindParam(':contact_number', $contactNumber);
        $updateStmt->bindParam(':full_name', $fullName);
        $updateStmt->bindParam(':investment', $investment);
        $updateStmt->bindParam(':percentage', $percentage);
        $updateStmt->bindParam(':mt5username', $user['mt5username']);
        
        // Execute the update query
        $updateStmt->execute();

        // Set success flag
        $updateSuccess = true;

        // Optionally, redirect to avoid form resubmission
        // header("Location: user_details.php?id=" . $userId);
        // exit;
        
        // Refresh the user details after update
        $stmt->execute();
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

    } catch (PDOException $e) {
        echo 'Update failed: ' . $e->getMessage();
    }
}

function formatOpenTrades($trades) {
    // Initialize an empty string to hold the formatted table rows
    $formattedTrades = '';

    // Split the trades by ';' to get individual trades
    $tradesArray = explode(';', $trades);

    // Iterate through each trade
    foreach ($tradesArray as $trade) {
        // Trim any extra spaces
        $trade = trim($trade);

        // Continue only if the trade is not empty
        if (!empty($trade)) {
            // Split each trade into fields by ',' 
            $fields = explode(',', $trade);

            // Initialize variables to hold field values
            $symbol = $volume = $openPrice = $stopLoss = $takeProfit = $profit = $type = '';

            // Iterate through each field and extract key-value pairs
            foreach ($fields as $field) {
                // Split the field into key and value by ':'
                list($key, $value) = explode(':', $field, 2);

                // Trim any extra spaces around key and value
                $key = trim($key);
                $value = trim($value);

                // Assign values to variables based on the key
                switch ($key) {
                    case 'Symbol':
                        $symbol = htmlspecialchars($value);
                        break;
                    case 'Volume':
                        $volume = htmlspecialchars($value);
                        break;
                    case 'Open Price':
                        $openPrice = htmlspecialchars($value);
                        break;
                    case 'Stop Loss':
                        $stopLoss = htmlspecialchars($value);
                        break;
                    case 'Take Profit':
                        $takeProfit = htmlspecialchars($value);
                        break;
                    case 'Profit':
                        $profit = htmlspecialchars($value);
                        break;
                    case 'Type':
                        $type = htmlspecialchars($value);
                        break;
                }
            }

            // Format the trade data into a table row
            $formattedTrades .= "<tr>
                                    <td>$symbol</td>
                                    <td>$stopLoss</td>
                                    <td>$takeProfit</td>
                                    <td>£$profit</td>
                                    <td>$type</td>
                                  </tr>";
        }
    }

    // If no trades were found, display a message
    if (empty($formattedTrades)) {
        $formattedTrades = '<tr><td colspan="5">No trades found or format issue.</td></tr>';
    }

    // Return the formatted table with headers
    return "<table>
                <thead>
                    <tr>
                        <th>Symbol</th>
                        <th>SL</th>
                        <th>TP</th>
                        <th>Profit</th>
                        <th>Type</th>
                    </tr>
                </thead>
                <tbody>$formattedTrades</tbody>
            </table>";
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Details</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600&display=swap" rel="stylesheet">
    <style>
        .container {
            width: 80%;
            margin-top: 15%;
            padding: 20px;
            background: #081012;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.5);
        }

        .section {
            margin-bottom: 20px;
            padding: 15px;
            background: #0f1819;
            border-radius: 5px;
        }

        .section h2 {
            color: #d4af37;
            margin-top: 0;
        }

        .section table {
            width: 100%;
            border-collapse: collapse;
            margin: 10px 0;
        }

        .section table th, .section table td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #444;
        }

        .section table thead {
            background-color: #1a2021;
            color: #d4af37;
        }

        .section table tr:nth-child(even) {
            background-color: #0f1819;
        }

        .section table tr:hover {
            background-color: #444;
        }

        .form-container {
            width: 50%;
            margin: 0 auto; /* Center the form horizontally */
            padding: 15px;
            background: #0f1819;
            border-radius: 5px;
        }

        .form-container h2 {
            color: #d4af37;
        }

        .form-container input, .form-container textarea {
            width: 50%;
            font-size: 16px; 
        }

        .form-container button {
            width: 50%;
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            background: #1a2021;
            color: #d4af37;
            font-size: 16px;
            cursor: pointer;
        }

        .form-container button:hover {
            background: #d4af37;
            color: #081012;
        }
/* Adjust font size and zoom for smaller screens */
@media only screen and (max-width: 600px) {
    body {
        font-size: 14px; /* Smaller font size for smaller screens */
        zoom: 0.9; /* Slightly zoom out to fit content */
        
    }
            .container {
                    zoom: 0.55; /* Slightly zoom out to fit content */

            }
}

    </style>
</head>
<body>
    <div class="container">
        <div class="section">
            <h2>User Details</h2>
            <table>
                <tr>
                    <td><strong>Username:</strong></td>
                    <td><?php echo htmlspecialchars($user['username']); ?></td>
                </tr>
                <tr>
                    <td><strong>Email:</strong></td>
                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                </tr>
                <tr>
                    <td><strong>Contact Number:</strong></td>
                    <td><?php echo htmlspecialchars($user['contact_number']); ?></td>
                </tr>
                <tr>
                    <td><strong>Full Name:</strong></td>
                    <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                </tr>
                <tr>
                    <td><strong>MT5 Username:</strong></td>
                    <td><?php echo htmlspecialchars($user['mt5username']); ?></td>
                </tr>
                <tr>
                    <td><strong>Investment:</strong></td>
                    <td>£<?php echo number_format($user['investment'], 2); ?></td>
                </tr>
                <tr>
                    <td><strong>Percentage:</strong></td>
                    <td><?php echo htmlspecialchars($user['percentage']); ?>%</td>
                </tr>
            </table>
        </div>

        <div class="section">
            <h2>Trade Information</h2>
            <?php echo formatOpenTrades($user['open_trades']); ?>
        </div>

        <div class="form-container">
            <h2>Edit User Details</h2>
            <?php if ($updateSuccess): ?>
                <p style="color: green;">User details updated successfully!</p>
            <?php endif; ?>
            <form method="POST" action=""><center>
                <label for="username">Username:</label><br>
                <input type="text" name="username" id="username" value="<?php echo htmlspecialchars($user['username']); ?>" required>
                <br>
                <label for="email">Email:</label><br>
                <input type="email" name="email" id="email" value="<?php echo htmlspecialchars($user['email']); ?>" required>
                <br>
                <label for="contact_number">Contact Number:</label><br>
                <input type="text" name="contact_number" id="contact_number" value="<?php echo htmlspecialchars($user['contact_number']); ?>" required>
                <br>
                <label for="full_name">Full Name:</label><br>
                <input type="text" name="full_name" id="full_name" value="<?php echo htmlspecialchars($user['full_name']); ?>" required>
                <br>
                <label for="investment">Investment:</label><br>
                <input type="number" step="0.01" name="investment" id="investment" value="<?php echo htmlspecialchars($user['investment']); ?>">
                <br>
                <label for="percentage">Percentage:</label><br>
                <input type="number" step="0.01" name="percentage" id="percentage" value="<?php echo htmlspecialchars($user['percentage']); ?>">
                <br>
                <button type="submit">Update Details</button></center>
            </form>
        </div>
    </div>
</body>
</html>
